////////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2009 The Regents of the University of California
//
// This file is part of Qbox
//
// Qbox is distributed under the terms of the GNU General Public License
// as published by the Free Software Foundation, either version 2 of
// the License, or (at your option) any later version.
// See the file COPYING in the root directory of this distribution
// or <http://www.gnu.org/licenses/>.
//
////////////////////////////////////////////////////////////////////////////////
//
// read.C: read a QSO potential file, build a DOM tree and write a UPF file
//
// for a description of the UPF format, see http://www.pwscf.org/format.txt
// for a description fo the QSO format, see http://www.quantum-simulation.org
//
// use: qso2upf file.xml file.upf
//
// input:  file.xml
// output: file.upf
//
////////////////////////////////////////////////////////////////////////////////

#include <libxml/parser.h>
#include <libxml/tree.h>
#include <cassert>
#include <iostream>
#include <iomanip>
#include <sstream>
#include <fstream>
#include <vector>
using namespace std;

double simpson(int n, const double *f, double h)
{
  // Simpson integral on n points
  const int np = ( n%2 == 0 ) ? n-1 : n;
  assert(np>=3);
  double sum = f[0] + 4.0 * f[1] + f[np-1];
  for ( int i = 2; i < np-2; i++, i++ )
    sum += 2.0 * f[i] + 4.0 * f[i+1];
  return sum * h / 3.0;
}
    
// compile with: g++ -I /usr/include/libxml2 -lxml2 -o qso2upf qso2upf.C
int main(int argc, char **argv)
{
  cerr << " qso2upf v1.2 2009-09-02" << endl;
  if ( argc != 3 )
  {
    cerr << " use: qso2upf file.xml file.upf" << endl;
    return 1;
  }
  int np; // size of radial grid
  int lmax, llocal;
  string description;
  string symbol;
  double mass;
  double zval;
  double dr;
  vector<vector<double> > v;
  vector<vector<double> > phi;
  vector<double> dij;

  xmlDoc *doc = 0;
  xmlNode *root_element = 0;

  LIBXML_TEST_VERSION

  doc = xmlReadFile(argv[1], 0, 0);

  if (doc == 0)
    cerr << " could not parse file " << argv[1] << endl;

  root_element = xmlDocGetRootElement(doc);

  // check that root element is <species>
  xmlNode *cur_node = root_element;
  assert(cur_node->type == XML_ELEMENT_NODE);
  assert(!xmlStrcmp(cur_node->name,(xmlChar*) "species"));

  // access children of <species> node
  cur_node = root_element->children;
  cur_node = cur_node->next;
 
  cout << " Analyzing XML DOM tree ..." << endl;
  // list all children of <species>
  do
  {
    // if ( cur_node->type == XML_ELEMENT_NODE )
    //   cout << cur_node->name << endl;
    if ( !xmlStrcmp(cur_node->name,(xmlChar*)"description") )
    {
      // process description element
      // cout << "processing <description> ... " << endl;
      xmlNode *p = cur_node->children;
      assert(p->type==XML_TEXT_NODE);
      description = (const char*) p->content;
      cout << description << endl;
      // cout << "done" << endl;
    }
    if ( !xmlStrcmp(cur_node->name,(xmlChar*)"symbol") )
    {
      // process symbol element
      // cout << "processing <symbol> ... ";
      xmlNode *p = cur_node->children;
      assert(p->type==XML_TEXT_NODE);
      cout << " symbol: " << p->content << endl;
      symbol = (const char*) p->content;
      // cout << "done" << endl;
    }
    if ( !xmlStrcmp(cur_node->name,(xmlChar*)"mass") )
    {
      // cout << "processing <mass> ... ";
      xmlNode *p = cur_node->children;
      assert(p->type==XML_TEXT_NODE);
      istringstream is_mass((const char*) p->content);
      is_mass >> mass;
      cout << " mass: " << mass << endl;
      // cout << "done" << endl;
    }
    if ( !xmlStrcmp(cur_node->name,(xmlChar*)"norm_conserving_pseudopotential"))
    {
      // cout << "processing <norm_conserving_pseudopotential> ... ";
      xmlNode *p = cur_node->children->next;

      // <valence_charge>
      assert(p->type==XML_ELEMENT_NODE);
      istringstream is_zval((const char*) p->children->content);
      is_zval >> zval;
      cout << " valence charge: " << zval << endl;
      p = p->next; 
      p = p->next; 

      // <lmax>
      assert(p->type==XML_ELEMENT_NODE);
      // cout << " element: " << p->name << endl;
      istringstream is_lmax((const char*) p->children->content);
      is_lmax >> lmax;
      cout << " lmax: " << lmax << endl;
      p = p->next; 
      p = p->next; 

      // <llocal>
      assert(p->type==XML_ELEMENT_NODE);
      // cout << " element: " << p->name << endl;
      istringstream is_llocal((const char*) p->children->content);
      is_llocal >> llocal;
      cout << " llocal: " << llocal << endl;
      p = p->next; 
      p = p->next; 

      // <nquad>
      assert(p->type==XML_ELEMENT_NODE);
      // cout << " element: " << p->name << endl;
      // skip: nquad is not used in UPF
      p = p->next; 
      p = p->next; 

      // <rquad>
      assert(p->type==XML_ELEMENT_NODE);
      // cout << " element: " << p->name << endl;
      // skip: rquad is not used in UPF
      p = p->next; 
      p = p->next; 

      // <mesh_spacing>
      assert(p->type==XML_ELEMENT_NODE);
      // cout << " element: " << p->name << endl;
      istringstream is_dr((const char*) p->children->content);
      is_dr >> dr;
      cout << " mesh spacing: " << dr << endl;
      p = p->next; 
      p = p->next; 

      xmlNode* pp = p;
      // pp points to the <projector> element

      v.resize(lmax+1);
      phi.resize(lmax+1);
      dij.resize(lmax+1);
      for ( int l = 0; l <= lmax; l++ )
      {

      p = pp;
      // <projector>
      assert(p->type==XML_ELEMENT_NODE);
      cout << " element: " << p->name << endl;

      // <projector l=".." >
      xmlAttr *pa = p->properties;
      assert(pa->type==XML_ATTRIBUTE_NODE);
      cout << " attribute: " << pa->name << endl;

      xmlNode *val = pa->children;
      assert(val->type==XML_TEXT_NODE);
      cout << " attribute value: " << val->content << endl;

      // next attribute: size
      pa = pa->next;
      assert(pa->type==XML_ATTRIBUTE_NODE);
      cout << " attribute: " << pa->name << endl;

      val = pa->children;
      assert(val->type==XML_TEXT_NODE);
      cout << " attribute value: " << val->content << endl;
      istringstream is((const char*) val->content);
      is >> np;
      v[l].resize(np);

      // contents of projector element
      // <radial_potential> element
      p = p->children->next;
      assert(p->type==XML_ELEMENT_NODE);
      cout << " " << p->name << endl;

      // contents of radial potential
      xmlNode *pc = p->children;
      const string s((const char*) pc->content);
      istringstream is2(s);
      for ( int i = 0; i < np; i++ )
        is2 >> v[l][i];
        
      // for ( int i = 0; i < 3; i++ )
      //   cout << v[l][i] << endl;

      // radial_function is present
      // insert here if (p != 0 )...
      p = p->next;
      p = p->next;

      // p now points to element <radial_function>
      if ( p != 0 )
      {
        assert(p->type==XML_ELEMENT_NODE);
        cout << " " << p->name << endl;

        phi[l].resize(np);
        // contents of radial function
        pc = p->children;
        //assert(p->type==XML_TEXT_NODE);
        const string s3((const char*) pc->content);
        istringstream is3(s3);
        for ( int i = 0; i < np; i++ )
          is3 >> phi[l][i];
        
        // for ( int i = 0; i < 3; i++ )
        //   cout << phi[l][i] << endl;
      }

      // go to the next <projector>
      pp = pp->next;
      pp = pp->next;

      } // for l
    }
    cur_node = cur_node->next;
  }
  while ( cur_node != 0 );
  
  xmlFreeDoc(doc);
  xmlCleanupParser();

  // process data
  // compute dij[l]
  for ( int l = 0; l <= lmax; l++ )
  {
    if ( l != llocal )
    {
      vector<double> f(np);
      // integral of r^2 phi_l^2 * (V_l - V_local) 
      double sum = 0.0;
#if 0
      for ( int i = 0; i < np; i++ )
        sum += dr * ( phi[l][i]*phi[l][i]*i*dr*i*dr*(v[l][i]-v[llocal][i]) );
      cout << "l=" << l << " sum=" << sum << endl;
#endif
      for ( int i = 0; i < np; i++ )
        f[i] = phi[l][i]*phi[l][i]*i*dr*i*dr*(v[l][i]-v[llocal][i]);
      sum = simpson(np,&f[0],dr);
      cout << " l=" << l << " sum=" << sum << endl;

      // norm: int r^2 phi^2 dr
      double nrm = 0.0;
#if 0
      for ( int i = 0; i < np; i++ )
        nrm += dr * ( phi[l][i]*phi[l][i]*i*dr*i*dr );
      cout << "l=" << l << " norm=" << nrm << endl;
#endif
      for ( int i = 0; i < np; i++ )
        f[i] = phi[l][i]*phi[l][i]*i*dr*i*dr;
      nrm = simpson(np,&f[0],dr); 
      cout << " l=" << l << " norm=" << nrm << endl;
      dij[l] = 0.5 * nrm / sum;
    }
  }

  cout << " XML DOM analysis done" << endl;

  // compute occupation numbers using simple filling rule
  vector<double> occ(lmax+1);
  double z = zval;
  for ( int l = 0; l <= lmax; l++ )
  {
    if ( l == 0 )
      occ[l] = min(2.0,z);
    else if ( l == 1 )
      occ[l] = min(6.0,z);
    else if ( l == 2 )
      occ[l] = min(10.0,z);

    z -= occ[l];
    cout << " occ[" << l << "] = " << occ[l] << endl;
  }

  // truncate array sizes for PWSCF limit of 2000 points
  np = min(np,2000);
      
  // write UPF file
  ofstream upf(argv[2]);
  upf << "<PP_INFO>" << endl;
  upf << "Translated from " << argv[1] << " using qso2upf" << endl;
  upf << description << endl;
  upf << "</PP_INFO>" << endl;

  upf << endl << endl;
  upf << "<PP_HEADER>" << endl;
  upf << "   0                   Version Number" << endl;
  upf << setw(4) << symbol << "                   Element" << endl;
  upf << "   NC                  Norm - Conserving pseudopotential" << endl;
  upf << "    F                  Nonlinear Core Corretion" << endl;
  upf << " XXX  CCC  XXX CCC     Exchange-Correlation functional" << endl;
  
  upf << setw(16) << fixed << setprecision(10) << zval 
      << "       Z valence" << endl;
  upf << "    0.00000000000      Total energy" << endl;
  upf << "  0.0000000  0.0000000 Suggested cutoff for wfc and rho" << endl;
  upf << setw(5) << lmax-1 << "                  Max angular momentum component"
      << endl;
  upf << setw(5) << np << "                  Number of points in mesh" << endl;
  upf << setw(5) << phi.size() << setw(5) << phi.size()-1
      << "             Number of Wavefunctions, Number of Projectors" << endl;
  upf << " Wavefunctions         nl  l   occ" << endl;
  //upf << "nl pn  l   occ               Rcut"
  //    << "            Rcut US             E pseu" << endl;
  for ( int l = 0; l <= lmax; l++ )
  {
    // write info on wavefunctions
    upf << " NL   " << l << " " << occ[l] << endl;
  }
  upf << "</PP_HEADER>" << endl;

  upf << endl << endl;
  upf << "<PP_MESH>" << endl;
  upf << "  <PP_R>" << endl;
  upf.setf(ios::scientific,ios::floatfield);
  for ( int i = 0; i < np; i++ )
  {
    upf << setw(19) << setprecision(11) << i*dr;
    if ( i % 4 == 3 )
      upf << endl;
  }
  if ( np % 4 != 0 )
    upf << endl;
  upf << "  </PP_R>" << endl;
  upf << "  <PP_RAB>" << endl;
  upf.setf(ios::scientific,ios::floatfield);
  for ( int i = 0; i < np; i++ )
  {
    upf << setw(19) << setprecision(11) << dr;
    if ( i % 4 == 3 )
      upf << endl;
  }
  if ( np % 4 != 0 )
    upf << endl;
  upf << "  </PP_RAB>" << endl;
  upf << "</PP_MESH>" << endl;
  
  upf << endl << endl;
  upf << "<PP_LOCAL>" << endl;
  upf.setf(ios::scientific,ios::floatfield);
  for ( int i = 0; i < np; i++ )
  {
    upf << setw(19) << setprecision(11) << 2.0*v[llocal][i];
    if ( i % 4 == 3 )
      upf << endl;
  }
  if ( np % 4 != 0 )
    upf << endl;
  upf << "</PP_LOCAL>" << endl;

  upf << endl << endl;
  upf << "<PP_NONLOCAL>" << endl;
  int ibeta = 0;
  for ( int l = 0; l <= lmax; l++ )
    if ( l != llocal )
    {
      ibeta++;
      upf << "  <PP_BETA>" << endl;
      upf << "  " << ibeta << "   " << l << "       Beta   L" << endl; 
      upf << "   " << np << endl;
      upf.setf(ios::scientific,ios::floatfield);
      for ( int i = 0; i < np; i++ )
      {
        upf << setw(19) << setprecision(11) 
            << 2.0*i*dr*phi[l][i]*(v[l][i]-v[llocal][i]);
        if ( i % 4 == 3 )
          upf << endl;
      }
      if ( np % 4 != 0 )
        upf << endl;
      upf << "  </PP_BETA>" << endl;
    }
  upf << "  <PP_DIJ>" << endl;
  upf << "  " << lmax << "           Number of nonzero Dij" << endl;
  int idij = 0;
  for ( int l = 0; l <= lmax; l++ )
  {
    if ( l != llocal )
    {
      idij++;
      upf << "    " << idij << " " << " " << idij << " " << dij[l] << endl;
    }
  }
  upf << "  </PP_DIJ>" << endl;
  upf << "</PP_NONLOCAL>" << endl;

  // write wavefunctions
  upf << endl << endl;
  upf << "<PP_PSWFC>" << endl;
  for ( int l = 0; l <= lmax; l++ )
  {
    upf << "NL  " << l << " " << occ[l] << " Wavefunction" << endl;
    for ( int i = 0; i < np; i++ )
    {
      upf << setw(19) << setprecision(11) << phi[l][i]*i*dr;
      if ( i % 4 == 3 )
        upf << endl;
    }
    if ( np % 4 != 0 )
      upf << endl;
  }
  upf << "</PP_PSWFC>" << endl;

  // compute total density
  vector<double> rho(np); 
  for ( int i = 0; i < np; i++ )
    rho[i] = 0.0;
  for ( int l = 0; l <= lmax; l++ )
  {
    for ( int i = 0; i < np; i++ )
      rho[i] += occ[l] * phi[l][i]*phi[l][i]*i*i*dr*dr;
  }    

  // write total density
  upf << endl << endl;
  upf << "<PP_RHOATOM>" << endl;
  for ( int i = 0; i < np; i++ )
  {
    upf << setw(19) << setprecision(11) << rho[i];
    if ( i % 4 == 3 )
      upf << endl;
  }
  if ( np % 4 != 0 )
    upf << endl;
  upf << "</PP_RHOATOM>" << endl;
    
  return 0;
}
