#!/usr/bin/env python3
#
# qbox_vavg.py:
# use: qbox_vavg.py file.cube
# generate plot data of averaged local potential in x,y,z directions
# read data from cube file generated by Qbox with the plot -vlocal command
# write plot data on standard output in gnuplot format
# current version works in orthorhombic cells only

import numpy as np
import sys

def usage():
  print("use: ",sys.argv[0]," cube_file")
  sys.exit()

argc=len(sys.argv)
if ( argc != 2 ):
  usage()

filename = sys.argv[1]

f = open(filename, 'r')
lines = f.readlines()

# number of atoms and origin on third line
buf = lines[2].split()
nat = int(buf[0])
origin = [ float(buf[1]), float(buf[2]), float(buf[3]) ]
#print("nat = ", nat)
#print("origin = ", origin)

# read grid parameters
buf = lines[3].split()
np0 = int(buf[0])
da0 = [ float(buf[1]), float(buf[2]), float(buf[3]) ]
#print(np0,da0)
buf = lines[4].split()
np1 = int(buf[0])
da1 = [ float(buf[1]), float(buf[2]), float(buf[3]) ]
#print(np1,da1)
buf = lines[5].split()
np2 = int(buf[0])
da2 = [ float(buf[1]), float(buf[2]), float(buf[3]) ]
#print(np2,da2)

# check that the cell is orthorhombic
assert da0[1] == 0.0 and da0[2] == 0
assert da1[0] == 0.0 and da1[2] == 0
assert da2[0] == 0.0 and da2[1] == 0

# read vlocal data starting at line 6+nat
v = np.zeros(np0*np1*np2)
pos = 0
for line in lines[6+nat:]:
  vals = ([float(val) for val in line.split()])
  length = len(vals)
  v[pos:pos+length] = vals[:]
  pos = pos + length

# fastest increasing index in cube file is z
v = v.reshape(np0,np1,np2)

vx = np.sum(v,(1,2))/(np1*np2)
vy = np.sum(v,(0,2))/(np0*np2)
vz = np.sum(v,(0,1))/(np0*np1)

#print("Vavg(x) min/max = ", '%8f %8f' %(min(vx), max(vx)))
#print("Vavg(y) min/max = ", '%8f %8f' %(min(vy), max(vy)))
#print("Vavg(z) min/max = ", '%8f %8f' %(min(vz), max(vz)))

# output position in bohr units
x = [origin[0]+da0[0]*i for i in range(np0)]
y = [origin[1]+da1[1]*i for i in range(np1)]
z = [origin[2]+da2[2]*i for i in range(np2)]

# print average potential on stdout in gnuplot format
print("# Vavg(x)")
for i in range(np0):
  print('%.8f %.8f' %(x[i], vx[i]))

print("\n\n# Vavg(y)")
for i in range(np1):
  print('%.8f %.8f' %(y[i], vy[i]))

print("\n\n# Vavg(z)")
for i in range(np2):
  print('%.8f %.8f' %(z[i], vz[i]))
