////////////////////////////////////////////////////////////////////////////////
//
//  power.C : power spectrum of time series
//
//  use: power [-w {h|b|w}] data.dat spectrum.dat npoles fmin fmax dt
//
//  -w {h|b|w}      use Hann or Bartlett or Welch windowing (optional)
//  data.dat        input: contains the atomic coordinates at all steps
//  spectrum.dat    output: contains the estimate of the power spectrum
//  npoles          number of poles used in the maximum entropy algorithm
//  fmin fmax       bounds of the power spectrum wanted (cm-1)
//  dt              time step of the simulation (a.u.)
//
////////////////////////////////////////////////////////////////////////////////

#include<iostream>
#include<iomanip>
#include<cstring>
#include<cstdlib>
#include<string>
#include<fstream>
#include<vector>
using namespace std;
#include"D3vector.h"

// Precision of the finite difference formula: 1 == 5-point formula
const bool five_pt_diff = true;

void window ( vector<double> &data, int wtype );
void pwr ( const vector<double> &data, vector<double> &spe, int npoles,
           double fmin, double fmax, double dt, double &xmstot );
void memcof(const vector<double> &data, vector<double> &d, double &xms);
double evlmem ( double fdt, const vector<double> &c, double xms);

int main(int argc, char **argv )
{
  const double cm1 = 1382642.0; // conversion of frequencies from cm-1 to a.u.
  const int nfreq = 1000; // number of frequencies
  int wintype = 0;
  double xmstot=0.0;

  // parse command line arguments
  // use: power [-w {h|b|w}] data.dat spectrum.dat npoles fmin fmax dt
  if ( !(argc == 7 || argc == 9) )
  {
    cerr << "use: power [-w {h|b|w}] data.dat spectrum.dat "
         << "npoles fmin fmax dt" << endl;
    return 1;
  }
  int iarg = 1;
  if ( !strcmp(argv[iarg],"-w") )
  {
    iarg++;
    // read window type
    // "-w h" or "-w b" or "-w w"
    if ( !strcmp(argv[iarg],"h") )
    {
      wintype = 1;
      cout << " Hann windowing" << endl;
    }
    else if ( !strcmp(argv[iarg],"b") )
    {
      wintype = 2;
      cout << " Bartlett windowing" << endl;
    }
    else if ( !strcmp(argv[iarg],"w") )
    {
      wintype = 3;
      cout << " Welch windowing" << endl;
    }
    else
    {
      cerr << " Unknown window function" << endl;
      return 1;
    }
    iarg++;
  }
  ifstream inFile( argv[iarg] ) ;
  if ( !inFile )
  {
    cerr << " Unable to open file " << argv[iarg] << endl;
    return 1;
  }
  iarg++;
  ofstream spectrumFile ( argv[iarg] ) ;
  if ( !spectrumFile )
  {
    cerr << " Unable to open file " << argv[iarg] << endl;
    return 1;
  }
  iarg++;
  const int npoles = atoi(argv[iarg]);
  iarg++;
  const double fmin_cm = atoi(argv[iarg]);
  iarg++;
  const double fmax_cm = atoi(argv[iarg]);
  iarg++;
  const double dt = atof(argv[iarg]);

  vector<double> spe(nfreq), spetot(nfreq,0.0);
  const double fmin = fmin_cm/cm1;
  const double fmax = fmax_cm/cm1;

  // read data
  vector<double> f;
  string str;
  while ( inFile )
  {
    double x;
    inFile >> x;
    getline(inFile,str);
    f.push_back(x);
  }

  const int nit = f.size();
  cout << " data size: " << nit << endl;
  vector<double> data(f.size());

  if ( !five_pt_diff )
  {
    data[0] = f[1]-f[0];
    data[nit-1] = f[nit-1]-f[nit-2];
    for ( int it = 1; it < nit-1; it++ )
      data[it] = 0.5 * ( f[it+1] - f[it-1] );
  }
  else
  {
    // df/di = ( - f(i+2) + 8 * f(i+1) - 8 * f(i-1) + f(i-2) ) / 12.0
    data[0] = f[1] - f[0];
    data[1] = 0.5 * ( f[2] - f[0] );
    data[nit-2] = 0.5 * ( f[nit-1] - f[nit-3] );
    data[nit-1] = f[nit-1] - f[nit-2];
    for ( int it = 2; it < nit-2; it++ )
      data[it]=( f[it-2] - 8 * f[it-1] + 8 * f[it+1] - f[it+2] ) / 12.0 ;
  }

  // compute spectrum only if data is non-zero
  double ssq = 0.0;
  for ( int it = 0; it < nit; it++ )
    ssq += data[it] * data[it];
  if ( ssq > 1e-10 )
  {
    // power returns the power spectrum in spe[0..nfreq-1]
    window(data,wintype);
    pwr(data,spe,npoles,fmin,fmax,dt,xmstot);
    for ( int i = 0; i < nfreq; i++ )
      spetot[i] += spe[i];
  }

  spectrumFile << "# " << nfreq << " " 
  << nit << " steps, "
  << npoles << " poles, "
  << "dt=" << dt << " ";
  if ( wintype == 0 ) spectrumFile << "no window";
  if ( wintype == 1 ) spectrumFile << "Hann window";
  if ( wintype == 2 ) spectrumFile << "Bartlett window";
  if ( wintype == 3 ) spectrumFile << "Welsh window";
  spectrumFile << endl;
  for ( int i = 0; i < nfreq; i++ )
  {
    double freq = fmin_cm + (fmax_cm-fmin_cm) * ( ( (double) i ) / nfreq );
    spectrumFile << " " << freq << " " << spetot[i] << endl;
  }

  cout << " Residual: " << xmstot << endl;
  return 0;
}

// data windowing 
void window ( vector<double> &data, int wtype )
{
  const int npts = data.size();
  if ( wtype == 0 ) return;
  else if ( wtype == 1 )
  {
    // Hann window 
    for ( int i = 0; i < npts; i++ )
    {
      double t = ( 2 * M_PI * i ) / npts;
      data[i] *= 0.5  - 0.5 * cos ( t );
    }
  }
  else if ( wtype == 2 )
  {
    // Bartlett window
    for ( int i = 0; i < npts; i++ )
    {
      double t = (i-0.5*npts)/(0.5*npts);
      data[i] *= 1.0 - fabs(t);
    }
  }
  else if ( wtype == 3 )
  {
    // Welch window
    for ( int i = 0; i < npts; i++ )
    {
      double t = (i-0.5*npts)/(0.5*npts);
      data[i] *= 1.0 - t*t;
    }
  }
}

void pwr ( const vector<double> &data, vector<double> &spe, int npoles,
           double fmin, double fmax, double dt, double &xmstot )
{
  // Evaluate power spectrum of data 

  vector<double> coeff(npoles);
  double xms;

  memcof( data, coeff, xms );
  xmstot += xms;

  //for ( int i = 0; i < coeff.size(); i++ )
  //  cout << " coeff[" << i << "] = " << coeff[i] << endl;

  const int nfreq = spe.size();
  for ( int i = 0; i < nfreq; i++ )
  {
    double f = fmin + (fmax-fmin) * ( ( (double) i ) / nfreq );
    spe[i] = evlmem ( f*dt, coeff, xms );
  }

}

void memcof(const vector<double> &data, vector<double> &d, double &xms)
{
  // input: data[i] i=0,..,n 
  // output: linear prediction coefficients d[i], i=0,..,m
  // output: mean square discrepancy xms
  const int n = data.size();
  // m: number of poles
  const int m = d.size();
  vector<double> wk1(n,0.0), wk2(n,0.0), wkm(m,0.0);
  double p = 0.0;
  for ( int j = 0; j < n; j++ )
    p += data[j]*data[j];
  xms = p / n;

  for ( int j = 0; j < n-1; j++ )
    wk1[j] = data[j];
  for ( int j = 1; j < n; j++ )
    wk2[j-1] = data[j];

  for ( int k = 0; k < m; k++ )
  {
    double num = 0.0, denom = 0.0;
    for ( int j = 0; j < n-k-1; j++ )
    {
      num += wk1[j] * wk2[j];
      denom += wk1[j]*wk1[j] + wk2[j]*wk2[j];
    }
    d[k] = 2.0 * num / denom;
    xms *= 1.0 - d[k]*d[k];
    for ( int i = 0; i < k; i++ )
      d[i] = wkm[i] - d[k] * wkm[k-i-1];
    if ( k == m-1 ) return;
    for ( int i = 0; i < k+1; i++ )
      wkm[i] = d[i];
    for ( int j = 0; j < n-k-2; j++ )
    {
      wk1[j] -= wkm[k] * wk2[j];
      wk2[j] = wk2[j+1] - wkm[k] * wk1[j+1];
    }
  }
  assert(false);
}

double evlmem ( double fdt, const vector<double> &c, double xms)
{
  // input: c[i], xms, fdt 
  // output: power spectrum estimate at fdt=f*delta

  double sumr = 1.0, sumi = 0.0;
  double wr = 1.0, wi = 0.0, wpr, wpi, theta;

  theta = 2.0 * M_PI * fdt;
  wpr = cos ( theta );
  wpi = sin ( theta );
  for ( int i = 0 ; i < c.size(); i++ )
  {
    double wtemp = wr;
    wr = wr * wpr - wi * wpi;
    wi = wi * wpr + wtemp * wpi;
    sumr -= c[i] * wr;
    sumi -= c[i] * wi;
  }
  return xms / ( sumr*sumr + sumi*sumi );
}
